using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Web.Mvc;
using System.Web.Mvc.Html;
using System.Web.Routing;

namespace HIPS.Web.Components.Web
{
    public static class HtmlHelperExtensions
    {

        public static MvcHtmlString Alert<T>(this HtmlHelper<T> helper, List<string> values, string alertClass, bool htmlEncode = true, string prefix = null)
        {
            string[] enumeratedValues = values.ToArray();

            // No values
            if (!enumeratedValues.Any())
            {
                return MvcHtmlString.Empty;
            }

            // Single value
            if (enumeratedValues.Count() == 1)
            {
                return Alert(helper, enumeratedValues[0], alertClass, htmlEncode, prefix);
            }

            /* Multiple Values */

            // Encodes
            alertClass = helper.AttributeEncode(alertClass);
            if (htmlEncode)
            {
                enumeratedValues = enumeratedValues.Select(helper.Encode).ToArray();
                prefix = helper.Encode(prefix);
            }
            
            const string wrapTemplate = @"<div class=""alert {0}""><ul>{1}</ul></div>";
            const string itemTemplate = @"<li>{0}</li>";
            const string itemPrefixTemplate = @"<li><b>{0}</b> {1}</li>";

            // Prefixed alert support
            if (!String.IsNullOrEmpty(prefix))
            {
                string prefixedContent = String.Join("\n", enumeratedValues.Select(v => String.Format(itemPrefixTemplate, prefix, v)));
                return new MvcHtmlString(String.Format(wrapTemplate, alertClass, prefixedContent));
            }

            // Standard alert
            string content = String.Join("\n", enumeratedValues.Select(v => String.Format(itemTemplate, v)));
            return new MvcHtmlString(String.Format(wrapTemplate, alertClass, content));
        }

        public static MvcHtmlString Alert<T>(this HtmlHelper<T> helper, string value, string alertClass, bool htmlEncode = true, string prefix = null)
        {
            // Check blank values
            if (String.IsNullOrEmpty(value))
            {
                return MvcHtmlString.Empty;
            }

            const string template = @"<p class=""alert {0}"">{1}<p>";
            const string prefixTemplate = @"<p class=""alert {0}""><b>{1}</b> {2}<p>";

            // Encodes
            alertClass = helper.AttributeEncode(alertClass);
            if (htmlEncode)
            {
                value = helper.Encode(value);
                prefix = helper.Encode(prefix);
            }

            // Prefixed alert support
            if (!String.IsNullOrEmpty(prefix))
            {
                return new MvcHtmlString(String.Format(prefixTemplate, alertClass, prefix, value)); 
            }

            // Standard alert
            return new MvcHtmlString(String.Format(template, alertClass, value));
        }

        public static MvcHtmlString StaticControl<T>(this HtmlHelper<T> helper, string value, bool htmlEncode = true)
        {
            const string template = @"<div class=""form-group"">
                                <div class=""form-control-wrap no-label"">
                                    <p class=""text-center form-control-static"">{0}</p>
                                </div>
                            </div>";

            if (htmlEncode)
            {
                value = helper.Encode(value);
            }

            return new MvcHtmlString(String.Format(template, value));
        }

        public static MvcHtmlString EditorDropDownFor<T, TValue>(this HtmlHelper<T> helper, Expression<Func<T, TValue>> expression, IEnumerable<SelectListItem> values)
        {
            return helper.EditorFor(expression, "DropDownList", new { Values = values });
        }

        public static MvcHtmlString EditorDropDownFor<T, TValue>(this HtmlHelper<T> helper, Expression<Func<T, TValue>> expression, IEnumerable<SelectListItem> values, string optionLabel)
        {
            return helper.EditorFor(expression, "DropDownList", new { Values = values, OptionLabel = optionLabel });
        }

        public static MvcHtmlString EditorDropDownFor<T, TValue>(this HtmlHelper<T> helper, Expression<Func<T, TValue>> expression, IEnumerable<SelectListItem> values, string optionLabel, object additionalViewData)
        {
            helper.ViewDataContainer.ViewData["Values"] = values;
            helper.ViewDataContainer.ViewData["OptionLabel"] = optionLabel;

            return helper.EditorFor(expression, "DropDownList", additionalViewData);
        }

        public static MvcHtmlString DisplayDropDownFor<T, TValue>(this HtmlHelper<T> helper, Expression<Func<T, TValue>> expression, IEnumerable<SelectListItem> values)
        {
            return helper.DisplayFor(expression, "DropDownList", new { Values = values });
        }

        public static MvcHtmlString DisplayDropDownFor<T, TValue>(this HtmlHelper<T> helper, Expression<Func<T, TValue>> expression, IEnumerable<SelectListItem> values, string optionLabel)
        {
            return helper.DisplayFor(expression, "DropDownList", new { Values = values, OptionLabel = optionLabel });
        }

        public static MvcHtmlString DisplayDropDownFor<T, TValue>(this HtmlHelper<T> helper, Expression<Func<T, TValue>> expression, IEnumerable<SelectListItem> values, string optionLabel, object additionalViewData)
        {
            helper.ViewDataContainer.ViewData["Values"] = values;
            helper.ViewDataContainer.ViewData["OptionLabel"] = optionLabel;

            return helper.DisplayFor(expression, "DropDownList", additionalViewData);
        }

        public static MvcHtmlString RenderFor<T, TValue>(this HtmlHelper<T> helper, bool editMode, Expression<Func<T, TValue>> expression)
        {
            if (editMode)
            {
                return helper.EditorFor(expression);
            }

            return helper.DisplayFor(expression);
        }

        public static MvcHtmlString RenderFor<T, TValue>(this HtmlHelper<T> helper, bool editMode, Expression<Func<T, TValue>> expression, string templateName)
        {
            if (editMode)
            {
                return helper.EditorFor(expression, templateName, templateName);
            }

            return helper.DisplayFor(expression, templateName, templateName);
        }

        public static MvcHtmlString RenderFor<T, TValue>(this HtmlHelper<T> helper, bool editMode, Expression<Func<T, TValue>> expression, object additionalViewData)
        {
            if (editMode)
            {
                return helper.EditorFor(expression, additionalViewData);
            }

            return helper.DisplayFor(expression, additionalViewData);
        }

        public static MvcHtmlString RenderFor<T, TValue>(this HtmlHelper<T> helper, bool editMode, Expression<Func<T, TValue>> expression, string templateName, object additionalViewData)
        {
            if (editMode)
            {
                return helper.EditorFor(expression, templateName, additionalViewData);
            }

            return helper.DisplayFor(expression, templateName, additionalViewData);
        }

        /// <summary>
        /// Returns an anchor (a element) that contains the virtual path of the specified action, with the link text determined based on the title associated with the route.
        /// </summary>
        /// <param name="helper">The HtmlHelper.</param>
        /// <param name="actionName">The name of the action.</param>
        /// <param name="routeValues">An object that contains the parameters for the route.</param>
        /// <returns>MvcHtmlString representing the anchor (a element).</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 January 2014">Initial version.</change>
        /// </history>
        public static MvcHtmlString ActionLinkWithRouteTitle(this HtmlHelper helper, string actionName, object routeValues)
        {
            return helper.ActionLinkWithRouteTitle(
                actionName,
                null,
                routeValues);
        }

        /// <summary>
        /// Returns an anchor (a element) that contains the virtual path of the specified action, with the link text determined based on the title associated with the route.
        /// </summary>
        /// <param name="helper">The HtmlHelper.</param>
        /// <param name="actionName">The name of the action.</param>
        /// <param name="controllerName">The name of the controller.</param>
        /// <param name="routeValues">An object that contains the parameters for the route.</param>
        /// <returns>MvcHtmlString representing the anchor (a element).</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 January 2014">Initial version.</change>
        /// </history>
        public static MvcHtmlString ActionLinkWithRouteTitle(this HtmlHelper helper, string actionName, string controllerName, object routeValues)
        {
            return helper.ActionLink(
                RouteTable.Routes.GetRouteTitle(actionName, controllerName, routeValues, helper.ViewContext.RequestContext, helper.ViewContext.ViewData.Model),
                actionName,
                controllerName,
                routeValues,
                null);
        }

        /// <summary>
        /// Gets a string that represents the route title for the current route.
        /// </summary>
        /// <param name="helper">The HtmlHelper.</param>
        /// <returns>MvcHtmlString representing the route title.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 January 2014">Initial version.</change>
        /// </history>
        public static MvcHtmlString RouteTitle(this HtmlHelper helper)
        {
            return MvcHtmlString.Create((helper.ViewContext.RouteData.Route as Route).GetTitle(helper.ViewContext.ViewData.Model));
        }

        /// <summary>
        /// Gets a string that represents the route title for a route identified via the specified parameters.
        /// </summary>
        /// <param name="helper">The HtmlHelper.</param>
        /// <param name="actionName">The name of the action.</param>
        /// <returns>MvcHtmlString representing the route title.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 January 2014">Initial version.</change>
        /// </history>
        public static MvcHtmlString RouteTitleForAction(this HtmlHelper helper, string actionName)
        {
            return helper.RouteTitleForAction(actionName, null, null);
        }

        /// <summary>
        /// Gets a string that represents the route title for a route identified via the specified parameters.
        /// </summary>
        /// <param name="helper">The HtmlHelper.</param>
        /// <param name="actionName">The name of the action.</param>
        /// <param name="controllerName">The name of the controller.</param>
        /// <param name="routeValues">An object that contains the parameters for the route.</param>
        /// <returns>MvcHtmlString representing the route title.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="18 January 2014">Initial version.</change>
        /// </history>
        public static MvcHtmlString RouteTitleForAction(this HtmlHelper helper, string actionName, string controllerName, object routeValues)
        {
            return MvcHtmlString.Create(RouteTable.Routes.GetRouteTitle(actionName, controllerName, routeValues, helper.ViewContext.RequestContext, helper.ViewContext.ViewData.Model));
        }

    }
}